/*
 * Decompiled with CFR 0.152.
 */
package io.github.mortuusars.exposure.world.level.storage;

import com.google.common.base.Preconditions;
import com.mojang.logging.LogUtils;
import io.github.mortuusars.exposure.network.Packets;
import io.github.mortuusars.exposure.network.packet.clientbound.ExposureDataChangedS2CP;
import io.github.mortuusars.exposure.network.packet.clientbound.ExposureDataResponseS2CP;
import io.github.mortuusars.exposure.util.UnixTimestamp;
import io.github.mortuusars.exposure.world.level.storage.ExpectedExposure;
import io.github.mortuusars.exposure.world.level.storage.ExposureData;
import io.github.mortuusars.exposure.world.level.storage.RequestedPalettedExposure;
import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.BiConsumer;
import java.util.function.Function;
import net.minecraft.server.MinecraftServer;
import net.minecraft.server.level.ServerPlayer;
import net.minecraft.util.StringUtil;
import net.minecraft.world.level.saveddata.SavedData;
import net.minecraft.world.level.storage.DimensionDataStorage;
import net.minecraft.world.level.storage.LevelResource;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;

public class ExposureRepository {
    public static final int EXPECTED_TIMEOUT_SECONDS = 60;
    public static final String EXPOSURES_DIRECTORY_NAME = "exposures";
    private static final Logger LOGGER = LogUtils.getLogger();
    protected final MinecraftServer server;
    protected final DimensionDataStorage dataStorage;
    protected final Path worldFolderPath;
    protected final Path exposuresFolderPath;
    protected final Map<ServerPlayer, Set<ExpectedExposure>> expectedExposures = new HashMap<ServerPlayer, Set<ExpectedExposure>>();

    public ExposureRepository(MinecraftServer server) {
        this.server = server;
        this.dataStorage = server.overworld().getDataStorage();
        this.worldFolderPath = server.getWorldPath(LevelResource.ROOT);
        this.exposuresFolderPath = this.worldFolderPath.resolve("data/exposures");
    }

    public List<String> getAllIds() {
        this.dataStorage.save();
        File folder = this.exposuresFolderPath.toFile();
        @Nullable File[] filesList = folder.listFiles();
        if (filesList == null) {
            return Collections.emptyList();
        }
        return Arrays.stream(filesList).filter(file -> file != null && file.isFile()).map(file -> com.google.common.io.Files.getNameWithoutExtension((String)file.getName())).toList();
    }

    public RequestedPalettedExposure load(@NotNull String id) {
        Preconditions.checkNotNull((Object)id, (Object)"id");
        Preconditions.checkArgument((!StringUtil.isBlank((String)id) ? 1 : 0) != 0, (Object)"Cannot load exposure: id is empty.");
        String name = "exposures/" + id;
        @Nullable ExposureData exposureData = (ExposureData)this.dataStorage.get(ExposureData.factory(), name);
        if (exposureData == null) {
            File filepath = this.exposuresFolderPath.resolve(id + ".dat").toFile();
            if (!filepath.exists()) {
                LOGGER.error("Exposure '{}' was not loaded. File '{}' does not exist.", (Object)id, (Object)filepath);
                return RequestedPalettedExposure.NOT_FOUND;
            }
            LOGGER.error("Exposure '{}' was not loaded. Check above messages for errors.", (Object)id);
            return RequestedPalettedExposure.CANNOT_LOAD;
        }
        return RequestedPalettedExposure.success(exposureData);
    }

    public void save(@NotNull String id, ExposureData data) {
        Preconditions.checkArgument((!StringUtil.isBlank((String)id) ? 1 : 0) != 0, (Object)"Cannot save exposure: id is null or empty.");
        if (this.ensureExposuresDirectoryExists()) {
            String saveDataName = "exposures/" + id;
            this.dataStorage.set(saveDataName, (SavedData)data);
            data.setDirty();
            Packets.sendToAllClients(new ExposureDataChangedS2CP(id));
        }
    }

    public void update(@NotNull String id, Function<ExposureData, ExposureData> updateFunction) {
        Preconditions.checkArgument((!StringUtil.isBlank((String)id) ? 1 : 0) != 0, (Object)"Cannot update exposure: id is null or empty.");
        this.load(id).getData().ifPresent(exposure -> {
            ExposureData updatedExposure = (ExposureData)((Object)((Object)updateFunction.apply((ExposureData)((Object)exposure))));
            if (!updatedExposure.equals(exposure)) {
                this.save(id, updatedExposure);
            }
        });
    }

    public boolean delete(String id) throws IOException {
        return Files.deleteIfExists(this.exposuresFolderPath.resolve(id + ".dat"));
    }

    public void expect(ServerPlayer player, String id, BiConsumer<ServerPlayer, String> onReceived) {
        Preconditions.checkArgument((!StringUtil.isBlank((String)id) ? 1 : 0) != 0, (Object)"id cannot be null or empty.");
        Set exposures = this.expectedExposures.computeIfAbsent(player, pl -> new HashSet());
        exposures.add(new ExpectedExposure(id, UnixTimestamp.Seconds.fromNow(60), onReceived));
    }

    public void expect(ServerPlayer player, String id) {
        this.expect(player, id, (pl, i) -> {});
    }

    public void handleClientRequest(ServerPlayer player, String id) {
        RequestedPalettedExposure result;
        if (StringUtil.isBlank((String)id)) {
            LOGGER.error("Null or empty id cannot be used to get an exposure data. Player: '{}'", (Object)player.getScoreboardName());
            result = RequestedPalettedExposure.INVALID_ID;
        } else {
            result = this.load(id);
        }
        Packets.sendToClient(new ExposureDataResponseS2CP(id, result), player);
    }

    public void receiveClientUpload(ServerPlayer player, String id, ExposureData exposure) {
        if (!this.validateUpload(player, id)) {
            return;
        }
        this.save(id, exposure);
        this.onExposureReceived(player, id);
        LOGGER.debug("Saved exposure '{}' uploaded by '{}'.", (Object)id, (Object)player.getScoreboardName());
    }

    public void clearExpectedExposuresTimedOutLongAgo() {
        for (Map.Entry<ServerPlayer, Set<ExpectedExposure>> exposures : this.expectedExposures.entrySet()) {
            AtomicInteger cleared = new AtomicInteger();
            exposures.getValue().removeIf(expected -> {
                if (expected.isTimedOut(UnixTimestamp.Seconds.now() - 60L)) {
                    cleared.getAndIncrement();
                    return true;
                }
                return false;
            });
            if (cleared.get() <= 0) continue;
            LOGGER.info("Cleared {} timed out expected exposures of player: '{}'", (Object)cleared.get(), (Object)exposures.getKey().getScoreboardName());
        }
    }

    protected boolean validateUpload(ServerPlayer player, String id) {
        if (StringUtil.isBlank((String)id)) {
            LOGGER.error("Null or empty id cannot be used to save captured exposure. Player: '{}'", (Object)player.getScoreboardName());
            return false;
        }
        @Nullable ExpectedExposure expectedExposure = this.expectedExposures.getOrDefault(player, Collections.emptySet()).stream().filter(ee -> ee.id().equals(id)).findFirst().orElse(null);
        if (expectedExposure == null) {
            LOGGER.error("Received unexpected upload from player '{}' with ID '{}'. Discarding.", (Object)player.getScoreboardName(), (Object)id);
            return false;
        }
        if (expectedExposure.isTimedOut(UnixTimestamp.Seconds.now())) {
            LOGGER.error("Received expected upload from player '{}' with ID '{}' - {}seconds later than expected. Discarding.", new Object[]{player.getScoreboardName(), id, UnixTimestamp.Seconds.now() - expectedExposure.timeoutAt()});
            return false;
        }
        return true;
    }

    protected void onExposureReceived(ServerPlayer player, String id) {
        this.expectedExposures.getOrDefault(player, Collections.emptySet()).removeIf(expectedExposure -> {
            if (expectedExposure.id().equals(id)) {
                expectedExposure.onReceived().accept(player, id);
                return true;
            }
            return false;
        });
    }

    protected boolean ensureExposuresDirectoryExists() {
        try {
            return Files.exists(this.exposuresFolderPath, new LinkOption[0]) || this.exposuresFolderPath.toFile().mkdirs();
        }
        catch (Exception e) {
            LOGGER.error("Failed to create exposure storage directory: {}", (Object)e.toString());
            return false;
        }
    }
}

